#!/system/bin/sh

# Replicant USB Networking
# ========================
#
# Copyright (C) 2011,2014 Paul Kocialkowski and Riku Saikkonen, GPLv3+
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

set -e

CFG_TYPE="static"               # default to "dhcp" or "static"
USB_IFACE="rndis0"
STATIC_IP="192.168.4.202"
STATIC_IFCONFIG_OPTIONS=""      # e.g. "netmask 255.255.255.0"
STATIC_GATEWAY="192.168.4.200"
STATIC_DNS1="8.8.8.8"
STATIC_DNS2=""

start_rndis () {
    #echo 1 >/sys/class/android_usb/android0/enable
    #setprop sys.usb.config rndis,adb
    svc usb setFunction rndis
}

stop_rndis () {
    #echo 0 >/sys/class/android_usb/android0/enable
    #setprop sys.usb.config mtp,adb
    svc usb setFunction
}

if_up () {
    #ifconfig $USB_IFACE up
    netcfg $USB_IFACE up
}

if_down () {
    #ifconfig $USB_IFACE down
    netcfg $USB_IFACE down
}

dns_changed () {
    # Signal possible change of DNS server to libc
    setprop net.dnschange $((`getprop net.dnschange`+1))
}

configure_static () {
    if_up
    ifconfig $USB_IFACE $STATIC_IP $STATIC_IFCONFIG_OPTIONS
    route add default gw $STATIC_GATEWAY dev $USB_IFACE
    setprop net.dns1 "$STATIC_DNS1"
    setprop net.dns2 "$STATIC_DNS2"
    dns_changed
    echo "Static IP $STATIC_IP configured"
}

configure_dhcp () {
    echo "Configuring DHCP, please wait"
    #if_up
    #dhcpcd -d $USB_IFACE
    netcfg $USB_IFACE dhcp
    setprop net.dns1 "`getprop net.$USB_IFACE.dns1`"
    setprop net.dns2 "`getprop net.$USB_IFACE.dns2`"
    dns_changed
    echo "DHCP configured"
}

configure () {
    case "$CFG_TYPE" in
        dhcp)
            configure_dhcp
            ;;
        static)
            configure_static
            ;;
        *)
            echo "$0: invalid type: $CFG_TYPE" 1>&2
            exit 1
            ;;
    esac
}

[ "" != "$2" ] && CFG_TYPE="$2"

case "$1" in
    start1)
        echo "Starting Replicant USB networking, phase 1"
        start_rndis
        ;;
    start2)
        echo "Starting Replicant USB networking, phase 2"
        configure
        ;;
    start)
        echo "Starting Replicant USB networking"
        start_rndis
        echo "Please start connection sharing on host now!"
        # work around short DHCP timeout by sleeping here
        if [ "dhcp" = "$CFG_TYPE" ]; then
            echo "Waiting 15 s"
            sleep 15
        fi
        configure
        ;;
    stop)
        echo "Stopping Replicant USB networking"
        if_down
        stop_rndis
        ;;
    *)
        echo "Usage: sh $0 {start|start1|start2|stop} [{dhcp|static}]"
        ;;
esac

exit 0
